<?php

class BalikobotAPIException extends Exception
{
    protected $messageList;

    public function __construct($messageList, $message = '', $code = 0, Exception $previous = null)
    {
        parent::__construct($message, $code, $previous);
        $this->messageList = $messageList;
    }

    public function getMessageList()
    {
        return $this->messageList;
    }
}

class BalikobotAPI
{
    const FUNCTION_ADD = 'add';
    const FUNCTION_DROP = 'drop';
    const FUNCTION_TRACK = 'track';
    const FUNCTION_TRACKSTATUS = 'trackstatus';
    const FUNCTION_LABELS = 'labels';
    const FUNCTION_ORDER = 'order';
    const FUNCTION_SERVICES = 'services';
    const FUNCTION_BRANCHES = 'branches';
    CONST FUNCTION_MANIPULATIONUNITS = 'manipulationunits';

    /** @var array Array of possible error codes returned by ADD method. */
    public static $addErrors = [
        '406' => [
            'eid' => 'Nedorazilo eshop ID.',
            'service_type' => 'Nedorazilo ID vybrané služby přepravce.',
            'cod_currency' => 'Nedorazil ISO kód měny.',
            'branch_id' => 'Nedorazilo ID pobočky.',
            'rec_name' => 'Nedorazilo jméno příjemce.',
            'rec_street' => 'Nedorazila ulice s číslem popisným příjemce.',
            'rec_city' => 'Nedorazilo město příjemce.',
            'rec_zip' => 'Nedorazilo PSČ příjemce.',
            'rec_country' => 'Nedorazil ISO kód země příjemce.',
            'rec_phone' => 'Nedorazilo telefonní číslo příjemce.',
            'rec_email' => 'Nedorazil email příjemce.',
            'price' => 'Nedorazila udaná cena zásilky.',
            'vs' => 'Nedorazil variabilní symbol pro dobírkovou zásilku.',
            'service_range' => 'Balíček nelze přidat, protože není vyplněna číselná řada v klientské zóně.',
            'config_data' => 'Balíček nelze přidat, protože chybí potřebná data v klientské zóně.',
        ],
        '413' => [
            'eid' => 'Eshop ID je delší než je maximální povolená délka.',
            'service_type' => 'Neznámé ID služby přepravce.',
            'cod_price' => 'Nepovolená dobírka.',
            'cod_currency' => 'Nepovolený ISO kód měny.',
            'branch_id' => 'Neznámé ID pobočky.',
            'rec_email' => 'Špatný formát emailu příjemce.',
            'order_number' => 'Sdružená zásilka není povolena.',
            'rec_country' => 'Nepovolený ISO kód země příjemce.',
            'rec_zip' => 'Nepovolené PSČ příjemce.',
            'swap' => 'Výměnná zásilka není pro vybranou službu povolena.',
            'rec_phone' => 'Špatný formát telefonního čísla.',
            'credit_card' => 'Platba kartou není pro tuto službu/pobočku povolena.',
            'service_range' => 'Balíček nelze přidat, protože číselná řada v klientské zóně je již přečerpaná.',
        ],
        '409' => [
            'cod_price + swap' => 'Nepovolená kombinace služeb dobírky a výměnné zásilky.',
        ],
    ];

    public static function add($data, $carrierCode, $apiCodes)
    {
        return self::makeAPIRequest(self::FUNCTION_ADD, $carrierCode, $data, $apiCodes);
    }

    public static function drop($packageId, $carrierCode, $apiCodes)
    {
        return self::makeAPIRequest(self::FUNCTION_DROP, $carrierCode, [ 'id' => $packageId ], $apiCodes);
    }

    public static function track($packageCarrierId, $carrierCode)
    {
        return self::makeAPIRequest(self::FUNCTION_TRACK, $carrierCode, [ 'id' => $packageCarrierId ]);
    }

    public static function trackStatus($packageCarrierId, $carrierCode)
    {
        return self::makeAPIRequest(self::FUNCTION_TRACKSTATUS, $carrierCode, [ 'id' => $packageCarrierId ]);
    }

    public static function labels($packageIds, $carrierCode, $apiCodes)
    {
        return self::makeAPIRequest(self::FUNCTION_LABELS, $carrierCode, [ 'package_ids' => $packageIds ], $apiCodes);
    }

    public static function order($data, $carrierCode, $apiCodes)
    {
        return self::makeAPIRequest(self::FUNCTION_ORDER, $carrierCode, $data, $apiCodes);
    }

    public static function services($carrierCode)
    {
        return self::makeAPIRequest(self::FUNCTION_SERVICES, $carrierCode);
    }

    public static function manipulationUnits($carrierCode)
    {
        return self::makeAPIRequest(self::FUNCTION_MANIPULATIONUNITS, $carrierCode);
    }

    public static function branches($carrierCode)
    {
        return self::makeAPIRequest(self::FUNCTION_BRANCHES, $carrierCode);
    }

    protected static function makeAPIRequest($function, $carrier, $data = null, $api_codes = null)
    {
        if(is_null($api_codes)){
            $api_codes = [
                'api_user' => Configuration::get('MONSTER_BALIKOBOT_API_USER'),
                'api_key' => Configuration::get('MONSTER_BALIKOBOT_API_KEY')
            ];
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api.balikobot.cz/" . $carrier . "/" . $function);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_HEADER, FALSE);
        if ($data != null) {
            curl_setopt($ch, CURLOPT_POST, TRUE);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER,
            array(
                "Authorization: Basic " . base64_encode($api_codes['api_user'] . ':' . $api_codes['api_key']), "Content-Type: application/json"
            )
        );
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $response = curl_exec($ch);
        curl_close($ch);

        if(Configuration::get('MONSTER_BALIKOBOT_DEBUG')){
            $logger = new FileLogger(0);
            $logger->setFilename(_PS_ROOT_DIR_ . "/log/mm_balikobot.log");
            $logger->logDebug("REQUEST: ".$carrier."/".$function." ".json_encode($data));
            $logger->logDebug("RESPONSE: ".$response);
            $logger->logDebug("------------------------------");
        }
        return json_decode($response, true);
    }
}
